import argparse
import os
import numpy as np
from evaluate.data_loader import load_truth_files
from experiment_runner import run_experiment, load_method

def main():
    parser = argparse.ArgumentParser(description='LogicBench Test Script')
    
    parser.add_argument(
        '-m', '--method', nargs='+', required=True, 
        help='Method names to run (refer to filenames in method folder). '
             'Multiple methods can be specified.'
    )
    parser.add_argument(
        '-d', '--data', nargs='+', required=True, 
        help='Data path and optional subdirectory filters. '
             'First argument is the base path (e.g., ANO). '
             'Additional arguments filter files by subdirectory names.'
    )
    parser.add_argument(
        '-p', '--problem', nargs='+', default=[], 
        help='Logic operators to use (and, or, not). '
    )
    parser.add_argument(
        '-n', '--num-files', type=int, default=-1, 
        help='Number of files to randomly sample from the dataset. '
             'Use -1 to process all files (default). '
             'Use positive integer to limit the number of files.'
    )
    parser.add_argument(
        '-s', '--split', type=float, default=0.75, 
        help='Train/test split ratio for data splitting. '
             'Default is 0.75 (75%% training, 25%% testing). '
             'For small datasets (≤128 samples), full data is used for both train and test.'
    )
    parser.add_argument(
        '-o', '--output-dir', default='output', 
        help='Output directory for saving results. '
             'Results are saved as JSON files in subdirectories mirroring the data structure.'
    )

    args = parser.parse_args()

    # Parse data path and subdirectory filters
    base_data_path = args.data[0]  # First parameter is base path
    subdir_filters = args.data[1:] if len(args.data) > 1 else [
    ]  # Remaining parameters are subdirectory filters

    # Load data files
    data_files = load_truth_files(os.path.join('data', base_data_path))
    if not data_files:
        print(f" No .truth files found in data/{base_data_path}")
        return

    # If subdirectory filters are specified, filter files
    if subdir_filters:
        filtered_files = []
        for file_path in data_files:
            # Check if file path contains any specified subdirectory filter
            if any(filter_name in file_path for filter_name in subdir_filters):
                filtered_files.append(file_path)
        data_files = filtered_files
        print(
            f" Filtered to {len(data_files)} files matching filters: {subdir_filters}"
        )

    if not data_files:
        print(f" No files found matching filters: {subdir_filters}")
        return

    # Randomly sample files if needed
    if args.num_files > 0 and args.num_files < len(data_files):
        data_files = np.random.choice(data_files,
                                      args.num_files,
                                      replace=False)

    print(f" Testing {len(data_files)} files from data/{base_data_path}")

    print(" Files to be processed:")
    print("-" * 60)
    for i, file_path in enumerate(data_files, 1):
        print(f"{i:3d}. {file_path}")
    print("-" * 60)

    # Run each method
    for method_name in args.method:
        print(f"\n Running method: {method_name}")
        operators = args.problem

        method = load_method(method_name)
        if not method:
            continue

        if hasattr(method, 'set_operators'):
            method.set_operators(operators)

        from evaluate.operator_config import set_method_operators
        set_method_operators(method_name, operators)

        run_experiment(method,
                       data_files,
                       split=args.split,
                       num_runs=1,
                       output_dir=args.output_dir,
                       method_name=method_name,
                       operators=operators)
        # Results are now saved automatically within run_experiment

if __name__ == '__main__':
    main()